/******************************************
Programme de calcul des OR par classes et de leur intervalle de confiance après une transformation de X par des fonctions splines


dernière mise à jour 28 août 2025 : JB


****************************************************
 Ce programme est appelé par la commande ORcl_sp après un modèle de régression logistique où une variable X est a été remplacée par des fonctions splines cubiques restreints avec 3 à 7 noeuds. Ces fonctions splines doivent avoir été créées par une des commandes makespline ou mkspline.
 
****************************************************
La syntaxe est : ORcl_sp varname,  [knots(numlist max=1)]  [knotslist(numlist max=7)] [knplace(string)] ref(real) cl(numlist min=1) [ajust(varlist fv)] 
Les arguments entre [ ] ne sont pas obligatoires

- varname est le nom de la variable transformée en splines

- [knots(numlist max=1)] est le nombre de noeuds
  [knotslist(numlist max=7)] est la liste des emplacements des noeuds (séparés par des blancs ou une virgules)
Une et une seule de ces options doit être fournie, celle qui a été donnée dans la commande initiale makespline (ou mkspline). Il y a une contrainte : il doit y avoir de 3 à 7 noeuds (c'est lié à la possibilité de l'option harrell)

- [knplace(string)] max=7)] n'est pas obligatoire. C'est la méthode de placements des noeuds qui peut être : harrell (les noeuds sont placés selon les recommandations de Harrell) ou uniformknots (les noeuds sont placés en des points équidistants sur l'étendue de X). Par défaut les noeuds sont placés aux percentiles de X.

- ref(#) est le centre de la classe de référence et # doit être remplacé par une seule valeur

- cl est la liste des centres de classes, séparés par des blancs ou des virgules, pour lesquelles on veut avoir l'OR et son IC (il peut n'y avoir qu'une seule classe).

- [(ajust)] n'est pas obligatoire. C'est la liste des variables d'ajustement dans le modèle logistique

Le programme se limite au cas où une seule variable X est transformée en splines
****************************************************/

/****************************************************
*****************************************************
Exemple
makespline rcs age, basis(vsp) knots(3) harrell replace 
logit acc `r(regressors)' ovo i.qual
ORcl_sp age, ref(27) cl(17 22 32 37 42) knots(3) knplace(harrell) ajust(ovo i.qual)


Résultat
OR et IC pour la variable age ajustée sur les variable ovo i.qual
Avec un modèle logistique où age est remplacé par 2 fonctions splines (dont la 1ère est age) : logit acc (2 variables splines) ovo i.qual

ref = 27   classe = 17   0R = 0.725    95% CI = [0.512 - 1.026]
ref = 27   classe = 22   0R = 0.851    95% CI = [0.716 - 1.013]
ref = 27   classe = 32   0R = 1.078    95% CI = [0.930 - 1.249]
ref = 27   classe = 37   0R = 0.663    95% CI = [0.555 - 0.792]
ref = 27   classe = 42   0R = 0.296    95% CI = [0.219 - 0.401]
*****************************************************
****************************************************/


capt program drop ORcl_sp
program define ORcl_sp

syntax varname,  [knots(numlist max=1)] ref(real) cl(numlist min=1) [ajust(varlist fv)] [knplace(string)] [knotslist(numlist max=7)]

***** 1. Enregistrement de la variable dépendante du modèle logistique
local y = "`e(depvar)'"

***** 2. Vérification des options sur les noeuds de la commande ORcl_sp 
if "`knots'" != "" & "`knotslist'" != "" {
	di in err  "Vous ne pouvez pas spécifier à la fois knots et knotslist. Veuillez en choisir un seul."
exit 198
}
if (wordcount("`knotslist'") < 3 & wordcount("`knotslist'") > 0)  {
	di in err "Il faut entre 3 et 7 nœuds"
	exit 198
}
if wordcount("`knotslist'")!=0 & "`knplace'"!="" {
	di in err "L'option knplace() n'est pas autorisée si la liste des noeuds est donnée"
	exit 198
}
if  "`knots'"!="" {
	if (`knots' < 3 | `knots' > 7) {
		di in err "Il faut entre 3 et 7 nœuds"
		exit 198
	}
}

***** 3. Classes successives de X dont on veut les OR. Elles sont nommées cli
local ncl : word count `cl'
tokenize `cl'
foreach i of numlist 1/`ncl' {
	local cl`i' ``i''
} 

***** 3. Valeurs des fonctions splines aux centres des classes successives de X dont on veut les OR
** dans la suite j indice les noeuds (ou les fonctions splines) et i les centres de classes

preserve   // pour ne pas changer le fichier d'origine

*** 3.1 Emplacements des noeuds obtenus par makespline sur le fichier d'origine
if "`knots'" !="" {
	qui makespline rcs `varlist', replace basis(_xxx_) knots(`knots') `knplace'
	local nds=""
	forvalues j = 1/`r(N_knots)' {
		local value`j' = r(knots)[1, `j']
		local nds ="`nds' " + "`value`j''"
		}
}
else {
	local nds `knotslist'
}


local nsp=`r(N_knots)'-2  // nb de fonctions splines autres que X

*** 2.1. Création d'un fichier avec une variable x contenant les valeurs des centres des classes sachant que makespline exige que le fichier de données ait au moins 10 valeurs différentes
clear
qui set obs `=`ncl'+11'  // au moins 10 sujets au total
qui gen x=`ref' in `=`ncl'+1'   // centre de la classe de référence
forvalues i = 1/`ncl' {
	qui replace x = `cl`i'' in `i'  // centres des autres classes
}
forvalues i = `=`ncl'+2'/`=`ncl'+11' {  // 10 valeurs pour satisfaire makespline
	qui replace x = `i' in `i'
}

*** 2.2. Construction des fonctions splines sur le nouveau fichier avec l'emplacements des noeuds enregistrés au $2.1
qui makespline rcs x, replace basis(vsp) knotslist(`nds') norescalevars


*** 2.3. Calcul des valeurs des fonctions splines aux centres des classes

forvalues i = 1/`ncl' {
	local  d1_`i'=x[`i']-x[`=`ncl'+1']

	forvalues k=1/`nsp' {
		local d2_`k'_`i'=vsp_1_`k'[`i']-vsp_1_`k'[`=`ncl'+1']
	}
}


restore  // on retourne au fichier d'origine


**** 5.Résultats

*** 5.1. Annonce des résultats
qui makespline rcs `varlist', replace basis(_xxx_) knotslist(`nds') `knotsplace' norescalevars

local nbaj : word count `ajust'
if `nbaj'==0 {
	local ajustement "non ajustée"
}
else if `nbaj'==1 {
	local ajustement "ajustée sur la variable `ajust'"
}
else {
	local ajustement "ajustée sur les variable `ajust'"
}

di as text _n(1) "OR et IC pour la variable `varlist' `ajustement', avec un modèle logistique où la variable `varlist' est remplacée par `=`knots'-1' fonctions splines (dont la 1ère est `varlist'). Le modèle est le suivant :"
di "logit acc (age remplacé par `=`knots'-1' variables splines) `ajust'" _n(1)

** 2. Calcul des OR et de leur IC 

*di "`r(regressors)'"
qui logit `y' `r(regressors)' `ajust'

local col : colnames e(b)  // récupération des noms des variables utilisées dans logit, utiles pour lincom
tokenize `col'
*disp "`col'"

local x1="`1'"
forvalue j = 1/`nsp' {
	local v`j' ="``=`j'+1''"
}

** 3. Ecriture du résultat (avec 3 décimales pour les OR et IC)
forvalues i = 1/ `ncl' {
	local linc`i'="`d1_`i''*`x1'"
	forvalues k=1/`nsp' {
		local linc`i' ="`linc`i'' + " + "`d2_`k'_`i''*`v`k''"
	}
	
qui lincom "`linc`i''",or

	
		disp as res "ref = " `ref' _s(3) "classe = " `cl`i'' ///
		_s(3) "0R = "  %-6.3f `r(estimate)' _s(3)  "`r(level)'% CI = [" %-6.3f `r(lb)' "-"  %6.3f `r(ub)' "]"
}		

drop _xxx_*

end


